#!/bin/bash

# TLS Version Support Test Script
# Tests TLS 1.2 and 1.3 connectivity for hosts listed in a CSV file
# Usage: ./testTlsVersionSupport.sh [input_csv_file] [output_csv_file]
#
# COMPATIBILITY: Designed for cross-platform use including Amazon Linux
# For compatibility check, run: ./checkCompatibility.sh
#
# INPUT FORMAT:
# CSV file with format: Component,Host,Port
# Example: Web Server,google.com,443
#
# OUTPUT VALUES:
# The script tests each host:port combination and returns one of these values for each TLS version:
#
# SUPPORTED     - TLS version is supported and working correctly
#                 The connection was established successfully using the specified TLS version
#
# NOT_SUPPORTED - SSL/TLS is supported but the specific TLS version is not available
#                 The server supports SSL/TLS but rejected the specific version (1.2 or 1.3)
#                 Common with legacy servers that only support older TLS versions (1.0/1.1)
#
# NON_SSL       - The port/service does not support SSL/TLS at all
#                 Connection succeeds but the service speaks plain-text protocols
#                 Examples: HTTP (port 80), FTP (port 21), SMTP (port 25/587)
#
# ERROR         - Connection or network failure prevented testing
#                 Includes: connection refused, timeouts, DNS failures, network unreachable
#                 These indicate infrastructure issues that need resolution before TLS testing
#
# UNTESTED      - Testing was intentionally skipped
#                 Currently only used for entries where port is -1
#
# OUTPUT FORMAT:
# CSV file with format: Component,Host,Port,TLS_1.2,TLS_1.3
# Example: Web Server,google.com,443,SUPPORTED,SUPPORTED

set -e

# Default file names
INPUT_CSV="${1:-tls_test_input.csv}"
OUTPUT_CSV="${2:-tls_test_results.csv}"

# Quick compatibility check
check_dependencies() {
    local missing_deps=()
    
    # Check essential commands
    for cmd in bash openssl timeout awk grep mktemp; do
        if ! command -v "$cmd" >/dev/null 2>&1; then
            missing_deps+=("$cmd")
        fi
    done
    
    if [[ ${#missing_deps[@]} -gt 0 ]]; then
        echo "Error: Missing dependencies: ${missing_deps[*]}"
        echo ""
        echo "For Amazon Linux 2: sudo yum install openssl"
        echo "For Amazon Linux 2023: sudo dnf install openssl"
        echo "Note: timeout command is available by default (part of coreutils)"
        echo ""
        echo "Run './checkCompatibility.sh' for detailed compatibility check"
        exit 1
    fi
}

# Perform compatibility check
check_dependencies

# Validate input file exists
if [[ ! -f "$INPUT_CSV" ]]; then
    echo "Error: Input CSV file '$INPUT_CSV' not found."
    echo "Usage: $0 [input_csv_file] [output_csv_file]"
    echo "Expected CSV format: Component,Host,Port"
    exit 1
fi

# Function to test TLS connectivity with Amazon Linux compatibility
test_tls_version() {
    local host="$1"
    local port="$2"
    local tls_version="$3"
    
    # Set timeout for connection attempt (using timeout command for Amazon Linux compatibility)
    local timeout_seconds=10
    local temp_error_file=$(mktemp)
    
    # Check if OpenSSL supports the requested TLS version flag
    if ! openssl s_client -help 2>&1 | grep -q "\-${tls_version}"; then
        rm -f "$temp_error_file"
        echo "ERROR"  # OpenSSL version doesn't support this TLS version flag
        return
    fi
    
    # Test TLS connectivity with specific version using timeout command
    if timeout "$timeout_seconds" openssl s_client -connect "${host}:${port}" "-${tls_version}" </dev/null >/dev/null 2>"$temp_error_file"; then
        rm -f "$temp_error_file"
        echo "SUPPORTED"
    else
        local exit_code=$?
        local error_output=$(cat "$temp_error_file" 2>/dev/null)
        rm -f "$temp_error_file"
        
        # Check if it's a timeout (exit code 124 for timeout command) or connection issue
        if [[ $exit_code -eq 124 ]]; then
            echo "ERROR"  # Timeout
        elif echo "$error_output" | grep -qi -E "(connection refused|network unreachable|name resolution|no route to host|connection timed out|connection reset|timeout|operation timed out|failed to connect|connect: connection refused|connect: network is unreachable)"; then
            echo "ERROR"  # Connection/network issues
        elif echo "$error_output" | grep -qi -E "(wrong version number|unknown protocol|ssl23_get_server_hello|not ssl|plaintext http|plain http|packet length too long|record layer failure|unexpected message|bad record mac|bad record type|decryption failed|record overflow)"; then
            echo "NON_SSL"  # Port doesn't support SSL/TLS at all
        elif echo "$error_output" | grep -qi -E "(handshake failure|protocol version|no protocols available|sslv3 alert|unsupported protocol|no shared cipher|no cipher overlap|certificate verify failed|tlsv1 alert protocol version)"; then
            echo "NOT_SUPPORTED"  # TLS version not supported but SSL is supported
        else
            # For other unclassified errors, default to ERROR for safety
            # (uncomment next line for debugging: echo "DEBUG: Unclassified error for $host:$port - Exit code: $exit_code, Error: $error_output" >&2)
            echo "ERROR"  # Unclassified errors default to ERROR
        fi
    fi
}

# Initialize output file with header
echo "Component,Host,Port,TLS_1.2,TLS_1.3" > "$OUTPUT_CSV"

echo "Starting TLS version support tests..."
echo "Input file: $INPUT_CSV"
echo "Output file: $OUTPUT_CSV"
echo "----------------------------------------"

# Skip the header line and process each row
tail -n +2 "$INPUT_CSV" | while IFS=',' read -r component host port; do
    # Remove any whitespace/carriage returns, but preserve internal spaces in component names
    component=$(echo "$component" | tr -d '\r\n\t' | sed 's/^[[:space:]]*//;s/[[:space:]]*$//')
    host=$(echo "$host" | tr -d '\r\n\t ' | sed 's/^[[:space:]]*//;s/[[:space:]]*$//')
    port=$(echo "$port" | tr -d '\r\n\t ' | sed 's/^[[:space:]]*//;s/[[:space:]]*$//')
    
    # Skip empty lines
    if [[ -z "$component" ]] || [[ -z "$host" ]]; then
        continue
    fi
    
    # Handle port -1 case (set as UNTESTED)
    if [[ "$port" == "-1" ]]; then
        echo "Marking $component ($host:$port) as UNTESTED - port is -1"
        tls12_result="UNTESTED"
        tls13_result="UNTESTED"
    else
        echo "Testing $component ($host:$port)..."
        
        # Test TLS 1.2
        echo "  Testing TLS 1.2..."
        tls12_result=$(test_tls_version "$host" "$port" "tls1_2")
        
        # Test TLS 1.3
        echo "  Testing TLS 1.3..."
        tls13_result=$(test_tls_version "$host" "$port" "tls1_3")
        
        echo "  Results: TLS 1.2=$tls12_result, TLS 1.3=$tls13_result"
    fi
    
    # Write results to output file
    echo "$component,$host,$port,$tls12_result,$tls13_result" >> "$OUTPUT_CSV"
    
    echo ""
done

echo "TLS testing completed!"
echo "Results saved to: $OUTPUT_CSV"

# Display summary
echo ""
echo "Summary of results:"
echo "==================="

# Count supported/not supported/untested/error/non-ssl for each TLS version
tls12_supported=$(awk -F',' 'NR>1 && $4=="SUPPORTED" {count++} END {print count+0}' "$OUTPUT_CSV")
tls12_not_supported=$(awk -F',' 'NR>1 && $4=="NOT_SUPPORTED" {count++} END {print count+0}' "$OUTPUT_CSV")
tls12_untested=$(awk -F',' 'NR>1 && $4=="UNTESTED" {count++} END {print count+0}' "$OUTPUT_CSV")
tls12_error=$(awk -F',' 'NR>1 && $4=="ERROR" {count++} END {print count+0}' "$OUTPUT_CSV")
tls12_non_ssl=$(awk -F',' 'NR>1 && $4=="NON_SSL" {count++} END {print count+0}' "$OUTPUT_CSV")
tls13_supported=$(awk -F',' 'NR>1 && $5=="SUPPORTED" {count++} END {print count+0}' "$OUTPUT_CSV")
tls13_not_supported=$(awk -F',' 'NR>1 && $5=="NOT_SUPPORTED" {count++} END {print count+0}' "$OUTPUT_CSV")
tls13_untested=$(awk -F',' 'NR>1 && $5=="UNTESTED" {count++} END {print count+0}' "$OUTPUT_CSV")
tls13_error=$(awk -F',' 'NR>1 && $5=="ERROR" {count++} END {print count+0}' "$OUTPUT_CSV")
tls13_non_ssl=$(awk -F',' 'NR>1 && $5=="NON_SSL" {count++} END {print count+0}' "$OUTPUT_CSV")

echo "TLS 1.2: $tls12_supported supported, $tls12_not_supported not supported, $tls12_untested untested, $tls12_error error, $tls12_non_ssl non-ssl"
echo "TLS 1.3: $tls13_supported supported, $tls13_not_supported not supported, $tls13_untested untested, $tls13_error error, $tls13_non_ssl non-ssl"

# Show the results file content
echo ""
echo "Results file content:"
echo "===================="
cat "$OUTPUT_CSV"
