#!/bin/bash

# TLS Version Support Script - Amazon Linux Compatibility Checker
# This script verifies all dependencies are available before running the main TLS test script

echo "=== TLS Version Support Script - Compatibility Check ==="
echo ""

# Color codes for output
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[0;33m'
NC='\033[0m' # No Color

# Track overall status
all_good=true

# Function to check if command exists
check_command() {
    local cmd="$1"
    local package="$2"
    local required="${3:-true}"
    
    if command -v "$cmd" >/dev/null 2>&1; then
        echo -e "${GREEN}✓${NC} $cmd - found at $(which "$cmd")"
        if [[ "$cmd" == "openssl" ]]; then
            local version=$(openssl version 2>/dev/null)
            echo "  Version: $version"
        fi
    else
        if [[ "$required" == "true" ]]; then
            echo -e "${RED}✗${NC} $cmd - NOT FOUND"
            if [[ -n "$package" ]]; then
                echo "  Install with: sudo yum install $package  (Amazon Linux 2)"
                echo "            or: sudo dnf install $package  (Amazon Linux 2023)"
            fi
            all_good=false
        else
            echo -e "${YELLOW}⚠${NC} $cmd - NOT FOUND (optional)"
        fi
    fi
}

# Function to check timeout command functionality
check_timeout_functionality() {
    if timeout 1 sleep 0.1 >/dev/null 2>&1; then
        echo -e "${GREEN}✓${NC} timeout command working correctly"
    else
        echo -e "${RED}✗${NC} timeout command not functioning properly"
        all_good=false
    fi
}

# Function to detect Amazon Linux version
detect_amazon_linux() {
    if [[ -f /etc/os-release ]]; then
        source /etc/os-release
        if [[ "$ID" == "amzn" ]]; then
            echo "Detected: $PRETTY_NAME"
            return 0
        fi
    fi
    
    if [[ -f /etc/system-release ]] && grep -q "Amazon Linux" /etc/system-release; then
        local version=$(cat /etc/system-release)
        echo "Detected: $version"
        return 0
    fi
    
    echo "Note: Not running on Amazon Linux (detected: ${ID:-Unknown})"
    return 1
}

echo "System Information:"
echo "=================="
detect_amazon_linux
echo ""

echo "Checking Core Dependencies:"
echo "=========================="
check_command "bash" "bash"
check_command "openssl" "openssl"
check_command "timeout" "coreutils"
check_command "awk" "gawk"
check_command "grep" "grep"
check_command "mktemp" "coreutils"
check_command "tail" "coreutils"
check_command "cat" "coreutils"
check_command "echo" "coreutils"
check_command "sed" "sed"
check_command "tr" "coreutils"
echo ""

echo "Checking Timeout Functionality:"
echo "==============================="
check_timeout_functionality
echo ""

echo "Testing OpenSSL TLS Support:"
echo "============================"
if command -v openssl >/dev/null 2>&1; then
    # Test if OpenSSL supports the TLS versions we need
    if openssl s_client -help 2>&1 | grep -q "tls1_2"; then
        echo -e "${GREEN}✓${NC} OpenSSL supports -tls1_2"
    else
        echo -e "${YELLOW}⚠${NC} OpenSSL may not support -tls1_2 flag"
        all_good=false
    fi
    
    if openssl s_client -help 2>&1 | grep -q "tls1_3"; then
        echo -e "${GREEN}✓${NC} OpenSSL supports -tls1_3"
    else
        echo -e "${YELLOW}⚠${NC} OpenSSL may not support -tls1_3 flag (older versions)"
        echo "  The script will still work but may not test TLS 1.3 properly"
    fi
else
    echo -e "${RED}✗${NC} OpenSSL not available - cannot test TLS support"
fi
echo ""

echo "Testing Basic Connectivity:"
echo "=========================="
# Test a simple connection to verify network and OpenSSL work together
if command -v openssl >/dev/null 2>&1; then
    echo "Testing connection to google.com:443..."
    if timeout 10 openssl s_client -connect google.com:443 -tls1_2 </dev/null >/dev/null 2>&1; then
        echo -e "${GREEN}✓${NC} Basic TLS connection test successful"
    else
        echo -e "${YELLOW}⚠${NC} Basic TLS connection test failed (network or firewall issue?)"
        echo "  The script may work for internal hosts but external tests might fail"
    fi
fi
echo ""

echo "Compatibility Summary:"
echo "====================="
if [[ "$all_good" == "true" ]]; then
    echo -e "${GREEN}✓ All dependencies satisfied!${NC}"
    echo "The TLS Version Support script should work without issues."
    echo ""
    echo "To run the main script:"
    echo "  ./testTlsVersionSupport.sh [input.csv] [output.csv]"
else
    echo -e "${RED}✗ Some dependencies are missing or have issues${NC}"
    echo "Please install missing packages before running the TLS test script."
    echo ""
    echo "For Amazon Linux 2:"
    echo "  sudo yum update"
    echo "  sudo yum install openssl"
    echo "  (timeout command is available by default in coreutils)"
    echo ""
    echo "For Amazon Linux 2023:"
    echo "  sudo dnf update"
    echo "  sudo dnf install openssl"
    echo "  (timeout command is available by default in coreutils)"
fi
echo ""

exit $([[ "$all_good" == "true" ]] && echo 0 || echo 1)



