package org.mule.tooling.troubleshooting.ui;

import java.util.Arrays;
import java.util.LinkedList;
import java.util.List;
import java.util.regex.Pattern;

import org.eclipse.core.resources.IProject;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.jface.layout.GridDataFactory;
import org.eclipse.jface.layout.GridLayoutFactory;
import org.eclipse.jface.viewers.CheckboxTreeViewer;
import org.eclipse.jface.viewers.ICheckStateListener;
import org.eclipse.jface.viewers.LabelProvider;
import org.eclipse.swt.SWT;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Group;
import org.eclipse.ui.dialogs.ContainerCheckedTreeViewer;
import org.mule.tooling.apidesigner.project.apiproject.ApiProjectHelper;
import org.mule.tooling.core.MuleCorePlugin;
import org.mule.tooling.core.model.IMuleProject;
import org.mule.tooling.core.model.MuleProjectKind;
import org.mule.tooling.core.utils.CoreUtils;
import org.mule.tooling.ui.MuleImages;
import org.mule.tooling.ui.widgets.tree.FullSelectionContainerCheckedTreeViewer;
import org.mule.tooling.ui.widgets.util.TextAndDecorationLabelProviderDecorator;

public class ProjectsChooserComponent {

    public static final Pattern PARENTHESIS_REGEX = Pattern.compile("(.+)(\\(.+\\))");
    public static final Pattern PROBLEMS_REGEX = Pattern.compile("(.+)(<.*)");
    private String title;
    private IProject[] projects;
    private CheckboxTreeViewer checkboxTreeViewer;

    public ProjectsChooserComponent(String title, IProject[] projects) {
        this.title = title;
        this.projects = projects;
    }

    public void createControl(Composite parent) {
        Group group = new Group(parent, SWT.NONE);
        group.setText(title);
        checkboxTreeViewer = createProjectsTree(group);

        GridDataFactory.fillDefaults().grab(true, true).applyTo(checkboxTreeViewer.getTree());
        GridLayoutFactory.swtDefaults().numColumns(1).applyTo(group);
        GridDataFactory.swtDefaults().align(SWT.FILL, SWT.CENTER).hint(SWT.DEFAULT, 200).grab(true, false).applyTo(group);

        checkboxTreeViewer.setInput(projects);
    }

    public void addCheckStateListener(ICheckStateListener listener) {
        checkboxTreeViewer.addCheckStateListener(listener);
    }

    public void setCheckedProjects(List<IMuleProject> selectedProjects) {
        checkboxTreeViewer.setCheckedElements(selectedProjects.toArray());
    }

    public List<IProject> getCheckedProjects() {
        List<IProject> selectedProjects = new LinkedList<>();
        Object[] checkedElements = checkboxTreeViewer.getCheckedElements();
        for (Object element : checkedElements) {

            selectedProjects.add((IProject) element);

        }
        return selectedProjects;
    }

    private CheckboxTreeViewer createProjectsTree(Composite parent) {
        final ContainerCheckedTreeViewer checkboxTreeViewer = new FullSelectionContainerCheckedTreeViewer(parent);
        checkboxTreeViewer.setAutoExpandLevel(CheckboxTreeViewer.ALL_LEVELS);
        checkboxTreeViewer.setContentProvider(new ProjectsContentProvider());

        checkboxTreeViewer.setLabelProvider(new TextAndDecorationLabelProviderDecorator(new LabelProvider() {

            @Override
            public String getText(Object element) {
                if (element instanceof IProject) {
                    IProject project = (IProject) element;
                    return project.getName();
                }
                return super.getText(element);
            }

            @Override
            public Image getImage(Object element) {

                if (element instanceof IProject) {
                    IProject project = (IProject) element;
                    try {
                        if (MuleProjectKind.APPLICATION.classifies(project)) {
                            return MuleImages.MULE_PROJECT_PACKAGE_EXPLORER;
                        } else if (MuleProjectKind.DOMAIN.classifies(project)) {
                            return MuleImages.DOMAIN_PROJECT_PACKAGE_EXPLORER;
                        } else if (ApiProjectHelper.isApiProject(project)) {
                            return MuleImages.API_PROJECT_PACKAGE_EXPLORER;
                        } else if (CoreUtils.hasSDKNature(project)) {
                            return MuleImages.SDK_PROJECT_PACKAGE_EXPLORER;
                        }
                    } catch (CoreException e) {
                        MuleCorePlugin.logError(e.getMessage(), e);

                    }
                }
                return super.getImage(element);
            }
        }, Arrays.asList(PROBLEMS_REGEX, PARENTHESIS_REGEX)));

        return checkboxTreeViewer;
    }

}
