package org.mule.tooling.troubleshooting.collectors;

import java.io.File;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;

import org.eclipse.core.runtime.IProgressMonitor;

public final class TroubleshootingCollectorsManager implements Iterable<Collector> {

    private Set<Collector> collectors;

    private TroubleshootingCollectorsManager() {
        collectors = new HashSet<>();
    }

    private static class DiagnosticCollectorsManagerHelper {

        private static final TroubleshootingCollectorsManager INSTANCE = new TroubleshootingCollectorsManager();
    }

    public static TroubleshootingCollectorsManager getInstance() {
        return DiagnosticCollectorsManagerHelper.INSTANCE;
    }

    public void registerCollector(Collector collector) {
        if (!collectors.add(collector)) {
            collectors.remove(collector);
            collectors.add(collector);
        }
    }

    public void unregisterCollector(Collector collector) {
        collectors.remove(collector);
    }

    public List<CollectorError> collect(File file, IProgressMonitor monitor) {
        Set<Collector> enabledCollectors = collectors.stream().filter(collector -> collector.isEnabled()).collect(Collectors.toSet());
        monitor.beginTask("Collect information", enabledCollectors.size());
        int progress = 0;
        List<CollectorError> errors = new ArrayList<>();
        for (Collector collector : enabledCollectors) {
            errors.addAll(collector.collect(file));
            monitor.worked(progress++);
        }
        collectors.clear();
        return errors;
    }

    public Iterator<Collector> iterator() {
        return collectors.iterator();
    }

    @Override
    public String toString() {
        return "TroubleshootingCollectorsManager [collectors=" + collectors + "]";
    }

}
