package org.mule.tooling.troubleshooting.collectors;

import java.io.File;
import java.io.IOException;
import java.lang.reflect.InvocationTargetException;
import java.util.ArrayList;
import java.util.List;

import org.apache.commons.io.FilenameUtils;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.mule.tooling.core.MuleRuntime;
import org.mule.tooling.core.builder.ApiProjectNature;
import org.mule.tooling.core.builder.MuleNature;
import org.mule.tooling.core.deployer.DefaultPackagingRequest;
import org.mule.tooling.core.deployer.DefaultPackagingRequest.Builder;
import org.mule.tooling.core.deployer.PackagingRequest;
import org.mule.tooling.core.io.MuleResourceUtils;
import org.mule.tooling.core.model.IMuleProject;
import org.mule.tooling.core.model.MuleProjectKind;
import org.mule.tooling.core.packageManager.BasePackageManager;
import org.mule.tooling.core.packageManager.IExportPackageManager;
import org.mule.tooling.core.packageManager.MuleExportStatus;
import org.mule.tooling.troubleshooting.ui.ProjectsSelectionPage;

public class CollectorProjects extends AbstractCollector {

    private static final String NAME = "Mule Projects";
    private static final String DESCRIPTION = "Select Mule Projects to be exported and added to zip file";
    private static final String ZIP_EXTENSION = ".zip";

    private ProjectsSelectionPage page;

    public CollectorProjects(ProjectsSelectionPage page) {
        super(page);
        this.page = page;
    }

    @Override
    public List<CollectorError> collect(File outputFolder) {
        File projects = new File(outputFolder, "exported-projects");
        projects.mkdir();
        List<CollectorError> out = new ArrayList<CollectorError>();
        for (IProject muleProject : page.getSelectedProjects()) {
            out.addAll(exportProject(muleProject, projects));
        }
        return out;
    }

    @Override
    public String getName() {
        return NAME;
    }

    @Override
    public String getDescription() {
        return DESCRIPTION;
    }

    private List<CollectorError> exportProject(IProject project, File outputExportPath) {
        boolean attachSources = true;
        boolean lightPackage = false;
        boolean exportParentDomain = false;
        boolean onlySources = false;

        List<CollectorError> out = new ArrayList<CollectorError>();

        try {
            if (project.hasNature(MuleNature.NATURE_ID)) {

                IMuleProject muleProject = MuleRuntime.create(project);
                muleProject.getServerDefinition();
                MuleExportStatus exportStatus = runExportProject(muleProject, outputExportPath, attachSources, lightPackage, exportParentDomain, onlySources);
                if (!exportStatus.isOk()) {
                    out.add(new CollectorError(this, "Project: " + muleProject.getName() + " has errors: " + exportStatus.getErrorMessage()));
                }
            } else if (project.hasNature(ApiProjectNature.NATURE_ID)) {
                File zipProject = MuleResourceUtils.zipProject(project);
                String projectName = FilenameUtils.removeExtension(zipProject.getName());
                File projectExportFile = new File(outputExportPath, projectName + ZIP_EXTENSION);
                zipProject.renameTo(projectExportFile);
            }
        } catch (CoreException | InterruptedException | InvocationTargetException | IOException e) {
            out.add(new CollectorError(this, "Error while exporting project: " + project.getName()));
        }

        return out;
    }

    private MuleExportStatus runExportProject(IMuleProject muleProject, File outputExportPath, boolean attachSources, boolean lightPackage, boolean exportParentDomain,
            boolean onlySources) throws InvocationTargetException, InterruptedException {

        MuleProjectKind<?> kind = muleProject.getKind();
        String extension = MuleProjectKind.APPLICATION.equals(kind) || MuleProjectKind.DOMAIN.equals(kind) ? BasePackageManager.APPLICATION_EXTENSION
                : BasePackageManager.ARCHIVE_DIR;

        File projectExportFile = new File(outputExportPath, muleProject.getName() + extension);

        Builder packageRequestBuilder = DefaultPackagingRequest.builder(muleProject).attachSources(attachSources).includeLinkedProjects(exportParentDomain)
                .outputFile(projectExportFile).withOnlySources(onlySources).withOption(IExportPackageManager.OPTIONS_SKIP_ALL_TESTS, true);

        PackagingRequest request = packageRequestBuilder.build();
        MuleExportStatus exportStatus = IExportPackageManager.get().export(request, new NullProgressMonitor());

        return exportStatus;
    }

}
