package org.mule.tooling.troubleshooting.collectors;

import java.io.File;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.List;

import org.apache.commons.io.FileUtils;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.Platform;
import org.mule.tooling.client.Activator;
import org.mule.tooling.client.startup.ToolingManager;
import org.mule.tooling.client.startup.ToolingNotAvailableException;
import org.mule.tooling.core.MuleCorePlugin;
import org.mule.tooling.core.deployer.MuleLocator;
import org.mule.tooling.core.packageManager.BasePackageManager;
import org.mule.tooling.core.runtime.server.IServerDefinition;
import org.mule.tooling.runtime.controller.IMuleInstance;
import org.mule.tooling.runtime.controller.MulePathHelper;
import org.osgi.framework.Bundle;

public class CollectorLogs extends AbstractCollector {

    private static final String NAME = "Tooling - Runtime - Workspace logs";
    private static final String DESCRIPTION = "Tooling, Runtime, Workspace  logs are added to zip file";
    private static final String RUNTIMES_FOLDER = ".runtimes";
    private static final String TOOLING_PREFIX = "tooling-";
    private String workspaceMetadataLogFile = ".metadata/.log";

    @Override
    public String getName() {
        return NAME;
    }

    @Override
    public String getDescription() {
        return DESCRIPTION;
    }

    @Override
    public List<CollectorError> collect(File outputFolder) {
        List<CollectorError> out = new ArrayList<CollectorError>();

        File runtimeDir = new File(outputFolder, "runtime-logs");
        runtimeDir.mkdir();

        MuleCorePlugin.getServerManager().getServerDefinitions().stream().forEach(definition -> {
            try {
                File runtimeX = new File(runtimeDir, definition.getVersionWithSuffix());
                copyRuntimeLogs(definition, runtimeX);
            } catch (IOException e) {
                out.add(new CollectorError(this, "[Runtime logs error] " + e.getMessage()));
            }
        });

        File workspaceDir = new File(outputFolder, "workspace-logs");
        workspaceDir.mkdir();

        try {
            copyWorkspaceLog(workspaceDir);
        } catch (IOException e) {
            out.add(new CollectorError(this, "[Workspace logs error] " + e.getMessage()));
        }

        File toolingDir = new File(outputFolder, "tooling-logs");
        toolingDir.mkdir();

        try {
            copyToolingLogs(toolingDir);
        } catch (IOException e) {
            out.add(new CollectorError(this, "[Tooling logs error] " + e.getMessage()));
        }

        return out;
    }

    private void copyRuntimeLogs(IServerDefinition definition, File directory) throws IOException {
        Path runtimePath = Paths.get(MuleLocator.create(definition).getInstallationDirectoryRootPath());
        MulePathHelper pathHelper = new MulePathHelper(runtimePath);
        final Path toolingLogsPath = pathHelper.getLogsPath();
        FileUtils.copyDirectory(toolingLogsPath.toFile(), directory);
    }

    private void copyWorkspaceLog(File directory) throws IOException {
        final String workspaceEclipseLogPath = ResourcesPlugin.getWorkspace().getRoot().getLocation().toOSString() + BasePackageManager.ROOT_PATH + workspaceMetadataLogFile;
        File workspaceLogFile = new File(directory, "workspace.log");
        FileUtils.copyFile(new File(workspaceEclipseLogPath), workspaceLogFile);
    }

    private void copyToolingLogs(File directory) throws IOException {

        // As tooling instance startup tries to remove old and unused instances, recover old instances should be done first.
        Bundle bundle = Platform.getBundle(Activator.PLUGIN_ID);
        File runtimesDirectory = bundle.getDataFile(RUNTIMES_FOLDER);
        File[] runtimes = runtimesDirectory.listFiles();
        if (runtimes != null) {
            for (File runtime : runtimes) {
                if (runtime.isDirectory() && runtime.getName().startsWith(TOOLING_PREFIX)) {
                    saveInstance(runtime.toPath(), runtime.getName(), directory);
                }
            }
        }

        List<String> lines = new ArrayList<>();
        lines.add("Running instance:");
        try {
            // Getting running instance.
            IMuleInstance instance = ToolingManager.getInstance().getToolingInstance();
            Path instancePath = instance.getPath();
            String instanceName = instancePath.getFileName().toString();
            lines.add(instanceName);

            // If running instance was started just in access above, tooling logs are added
            if (!folderContainsFolder(directory, instanceName)) {
                saveInstance(instancePath, instanceName, directory);
            }
        } catch (ToolingNotAvailableException e) {
            lines.add("NO RUNNING INSTANCE");
        }
        File file = new File(directory, "instance-status.log");
        Files.write(file.toPath(), lines, StandardCharsets.UTF_8);
    }

    private static void saveInstance(Path instancePath, String instanceName, File directory) throws IOException {
        MulePathHelper pathHelper = new MulePathHelper(instancePath);
        final Path toolingLogsPath = pathHelper.getLogsPath();

        File toolingInstanceDir = new File(directory, instanceName);
        toolingInstanceDir.mkdir();
        FileUtils.copyDirectory(toolingLogsPath.toFile(), toolingInstanceDir);
    }

    private static boolean folderContainsFolder(File directory, String folderName) {
        for (File toolingInstanceDir : directory.listFiles()) {
            if (toolingInstanceDir.isDirectory() && toolingInstanceDir.getName().equals(folderName)) {
                return true;
            }
        }
        return false;
    }

}
