package org.mule.tooling.troubleshooting;

import java.io.File;
import java.io.IOException;
import java.lang.reflect.InvocationTargetException;
import java.nio.file.FileSystems;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.List;

import org.apache.commons.io.FileUtils;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.operation.IRunnableWithProgress;
import org.eclipse.swt.widgets.Display;
import org.mule.tooling.core.io.MuleResourceUtils;
import org.mule.tooling.core.utils.InstallationInformation;
import org.mule.tooling.troubleshooting.collectors.CollectorError;
import org.mule.tooling.troubleshooting.collectors.TroubleshootingCollectorsManager;
import org.mule.tooling.troubleshooting.ui.DialogWithLink;
import org.mule.tooling.troubleshooting.ui.TroubleshootingExportWizard;
import org.mule.tooling.ui.widgets.util.WidgetUtils;

public class CreateTroubleshootingRunnable implements IRunnableWithProgress {

    private String outputFolderName = "troubleshootingDirectory";
    private static final String ZIP_EXTENSION = ".zip";
    private static String OS = System.getProperty("os.name");
    private static final String ZIP_FILE_NAME_HEADER = "diagnostic";

    private String outputZipFileName;

    private File folder;

    public CreateTroubleshootingRunnable(File folder) {
        super();
        this.folder = folder;
    }

    @Override
    public void run(IProgressMonitor monitor) throws InvocationTargetException, InterruptedException {
        TroubleshootingCollectorsManager collector = TroubleshootingCollectorsManager.getInstance();

        File out = new File(folder, outputFolderName);
        out.mkdir();

        List<CollectorError> errors = collector.collect(out, monitor);
        monitor.done();

        boolean zipAvaliable = false;
        try {
            outputZipFileName = getZipFileName();
            File destZip = new File(out.getParent(), outputZipFileName);
            MuleResourceUtils.zipDirectory(out, destZip);
            zipAvaliable = true;
        } catch (IOException e) {
            openTroubleshootingError("Error while zipping all resources. The situation is worst than expected");
        } finally {
            try {
                FileUtils.deleteDirectory(out);
            } catch (IOException e) {
                openTroubleshootingError("Error deleting temp logs file ");
            }
        }

        if (!errors.isEmpty()) {
            StringBuilder msj = new StringBuilder();
            for (CollectorError collectorError : errors) {
                msj.append(collectorError.getCollectorName());
                msj.append(": ");
                msj.append(collectorError.getMessage());
                msj.append("\n");
            }
            if (zipAvaliable) {
                finishAction(msj.toString());
            } else {
                openTroubleshootingError(msj.toString());
            }
        } else {
            finishAction(null);
        }

    }

    private static String getZipFileName() {
        String date = new SimpleDateFormat("yyyyMMdd_hhmmss").format(new Date());
        String os = "";
        if (OS.startsWith("Windows")) {
            os = "win";
        } else if (OS.startsWith("Linux")) {
            os = "linux";
        } else if (OS.startsWith("Mac")) {
            os = "osx";
        }

        return ZIP_FILE_NAME_HEADER + "_" + os + "_" + InstallationInformation.getFullStudioVersion().replace(".", "_") + "_" + date + ZIP_EXTENSION;
    }

    private void openTroubleshootingError(String errorMessage) {
        final Display display = WidgetUtils.getDisplay();
        display.asyncExec(new Runnable() {

            @Override
            public void run() {
                MessageDialog.openError(display.getActiveShell(), "Error while exporting: \n", errorMessage);
            }

        });
    }

    private void finishAction(final String errors) {
        final Display display = WidgetUtils.getDisplay();
        String separator = FileSystems.getDefault().getSeparator();
        display.asyncExec(new Runnable() {

            @Override
            public void run() {
                if (errors == null) {
                    String message = "Exported resources successfully in: " + folder.getAbsolutePath() + separator + outputZipFileName;
                    DialogWithLink.openInformation(display.getActiveShell(), TroubleshootingExportWizard.TITLE + " Finished", message, folder);
                } else {
                    String message = "Some collectors has errors: \n" + errors + "\n" + "Exported resources are in: " + folder.getAbsolutePath() + separator + outputZipFileName;
                    DialogWithLink.openWarning(display.getActiveShell(), TroubleshootingExportWizard.TITLE + " Finished", message, folder);
                }
            }

        });
    }

}
