@echo off
setlocal enabledelayedexpansion

rem Mule Troubleshooting Tool - Windows Batch Version

rem Jump to main execution to skip function definitions
goto :main

:printHelp
echo Mule Troubleshooting Tool
echo =========================
echo.
echo Usage: diag.bat [options] [command] [command-options]
echo.
echo Commands:
echo   diaf                      Generate a complete Mule diagnostic dump (default)
echo   help                      Show this help message
echo   ^<operation-name^>          Execute a specific troubleshooting operation
echo.
echo Global Options:
echo   --output ^<path^>           Specify custom output directory or file path
echo   --extended                Enable extended mode (includes heap dump)
echo   --debug                   Enable debug mode with remote debugging on port 5005
echo.
echo Examples:
echo   diag.bat                             # Generate diagnostic dump to logs directory
echo   diag.bat --extended                  # Include heap dump in diagnostic
echo   diag.bat --output C:\temp\mule.zip   # Save to specific file
echo   diag.bat --output C:\temp\           # Save to specific directory
echo   diag.bat ^<operation-name^>            # Execute specific operation
echo.
echo Output:
echo   By default, the tool creates a ZIP file containing:
echo   - mule_dump_^<timestamp^>.diaf       # Diagnostic information
echo   - thread_dump_^<timestamp^>.txt      # Thread dump
echo   - heap_dump_^<timestamp^>.hprof      # Heap dump (if --extended is used)
echo.
echo   The ZIP file is saved to the 'logs' directory by default.
goto :eof

:main
rem Check for help option
if "%1"=="--help" (
    call :printHelp
    exit /b 0
)
if "%1"=="-h" (
    call :printHelp
    exit /b 0
)
if "%1"=="help" (
    call :printHelp
    exit /b 0
)

rem Store original arguments and script directory before any shifts
set TEMP_ARGS=%*
set SCRIPT_DIR_ORIGINAL=%~dp0
rem Remove trailing backslash for consistent path handling
if "!SCRIPT_DIR_ORIGINAL:~-1!"=="\" set SCRIPT_DIR_ORIGINAL=!SCRIPT_DIR_ORIGINAL:~0,-1!

rem Parse command line arguments to find --output parameter
set OUTPUT_DIR=
set REMAINING_ARGS=

:parseArgs
set CURRENT_ARG=%1
if "!CURRENT_ARG!"=="" goto :argsParsed

if "!CURRENT_ARG!"=="--stdout" (
    echo. 1>&2
    echo ERROR: --stdout option is not supported on Windows due to console binary data limitations. 1>&2
    echo. 1>&2
    echo Please use one of these alternatives instead: 1>&2
    echo   diag.bat --output .\mule.zip    # Creates mule.zip 1>&2
    echo   diag.bat --output .\            # Saves files to current directory 1>&2
    echo   diag.bat                        # Uses default logs directory 1>&2
    echo. 1>&2
    exit /b 1
)

if "!CURRENT_ARG!"=="--output" (
    shift
    goto :getOutputValue
)

rem Check for --output=value format
echo !CURRENT_ARG! | findstr /c:"--output=" >nul
if !errorlevel! equ 0 (
    set OUTPUT_DIR=!CURRENT_ARG!
    set OUTPUT_DIR=!OUTPUT_DIR:--output=!
    shift
    goto :parseArgs
)

rem Add to remaining arguments
if "!REMAINING_ARGS!"=="" (
    set REMAINING_ARGS=!CURRENT_ARG!
) else (
    set REMAINING_ARGS=!REMAINING_ARGS! !CURRENT_ARG!
)
shift
goto :parseArgs

:getOutputValue
set CURRENT_ARG=%1
if "!CURRENT_ARG!"=="" (
    echo Error: --output requires a value 1>&2
    exit /b 1
)
set OUTPUT_DIR=!CURRENT_ARG!
shift
goto :parseArgs

:argsParsed

rem If no output directory specified, use default logs directory
if "!OUTPUT_DIR!"=="" (
    rem Use the original script directory captured before any shifts
    set SCRIPT_DIR=!SCRIPT_DIR_ORIGINAL!
    set OUTPUT_DIR=!SCRIPT_DIR!\..\logs
)

rem Determine output strategy based on intended output type
set OUTPUT_IS_ZIP=false
set ACTUAL_OUTPUT_DIR=
set TARGET_ZIP_PATH=

rem Convert forward slashes to backslashes for Windows
set OUTPUT_DIR=!OUTPUT_DIR:/=\!

rem Check if OUTPUT_DIR ends with backslash (or forward slash)
if "!OUTPUT_DIR:~-1!"=="\" (
    rem OUTPUT_DIR ends with \, treat as directory - use directly
    set ACTUAL_OUTPUT_DIR=!OUTPUT_DIR:~0,-1!
    set OUTPUT_IS_ZIP=false
    echo Using directory output mode: !ACTUAL_OUTPUT_DIR! 1>&2
) else (
    rem Check if OUTPUT_DIR is an existing directory using simple batch commands
    if exist "!OUTPUT_DIR!" (
        rem Check if it's a directory by trying to list it
        dir "!OUTPUT_DIR!" >nul 2>&1
        if !errorlevel! equ 0 (
            rem OUTPUT_DIR is an existing directory - use directly
            set ACTUAL_OUTPUT_DIR=!OUTPUT_DIR!
            set OUTPUT_IS_ZIP=false
            echo Using existing directory: !ACTUAL_OUTPUT_DIR! 1>&2
        ) else (
            rem It exists but is not a directory (probably a file) - use temporary directory for zip
            set OUTPUT_IS_ZIP=true
            for /f "delims=" %%i in ('powershell -command "[System.IO.Path]::GetTempPath() + [System.Guid]::NewGuid().ToString()"') do set ACTUAL_OUTPUT_DIR=%%i
            set TARGET_ZIP_PATH=!OUTPUT_DIR!
            echo Using temporary directory for zip output: !ACTUAL_OUTPUT_DIR! 1>&2
            echo Target zip file: !TARGET_ZIP_PATH! 1>&2
        )
    ) else (
        rem OUTPUT_DIR doesn't exist - assume it's a zip file path
        set OUTPUT_IS_ZIP=true
        for /f "delims=" %%i in ('powershell -command "[System.IO.Path]::GetTempPath() + [System.Guid]::NewGuid().ToString()"') do set ACTUAL_OUTPUT_DIR=%%i
        set TARGET_ZIP_PATH=!OUTPUT_DIR!
        echo Using temporary directory for zip output: !ACTUAL_OUTPUT_DIR! 1>&2
        echo Target zip file: !TARGET_ZIP_PATH! 1>&2
    )
)

rem Create actual output directory if it doesn't exist
if not exist "!ACTUAL_OUTPUT_DIR!" (
    echo Creating output directory: !ACTUAL_OUTPUT_DIR! 1>&2
    mkdir "!ACTUAL_OUTPUT_DIR!" 2>nul
    if !errorlevel! neq 0 (
        echo Error: Failed to create output directory !ACTUAL_OUTPUT_DIR!. Skipping diag. 1>&2
        exit /b 1
    )
)

rem Also ensure parent directory of zip file exists if needed
if "!OUTPUT_IS_ZIP!"=="true" (
    if not "!TARGET_ZIP_PATH!"=="" (
    for %%f in ("!TARGET_ZIP_PATH!") do set TARGET_ZIP_DIR=%%~dpf
    if not exist "!TARGET_ZIP_DIR!" (
        echo Creating target zip directory: !TARGET_ZIP_DIR! 1>&2
        mkdir "!TARGET_ZIP_DIR!" 2>nul
        if !errorlevel! neq 0 (
            echo Error: Failed to create target zip directory !TARGET_ZIP_DIR!. Skipping diag. 1>&2
            exit /b 1
        )
    )
    )
)

rem Set default MULE_BASE if not provided
if "!MULE_BASE!"=="" (
    set MULE_BASE=C:\mule\
    echo MULE_BASE not set, using default: !MULE_BASE! 1>&2
)

rem Setup Java environment
call :setupJava
if !errorlevel! neq 0 exit /b !errorlevel!

rem Call muleTroubleshooting function
call :muleTroubleshooting "!ACTUAL_OUTPUT_DIR!" "!REMAINING_ARGS!"
set JAVA_EXIT_CODE=!errorlevel!

rem Handle zipping if needed
if "!OUTPUT_IS_ZIP!"=="true" (
    echo Creating zip from generated files... 1>&2
    call :createZipFile "!ACTUAL_OUTPUT_DIR!" "!TARGET_ZIP_PATH!"
    
    rem Clean up temporary directory
    echo Cleaning up temporary directory: !ACTUAL_OUTPUT_DIR! 1>&2
    rmdir /s /q "!ACTUAL_OUTPUT_DIR!" 2>nul
) else (
    echo Files generated in directory: !ACTUAL_OUTPUT_DIR! 1>&2
)

exit /b !JAVA_EXIT_CODE!

:setupJava
rem Setup JAVA_HOME if not already set
if "!JAVA_HOME!"=="" (
    rem Try to find Java from PATH
    for /f "delims=" %%i in ('where java 2^>nul') do (
        set JAVA_PATH=%%i
        for %%j in ("!JAVA_PATH!") do set JAVA_DIR=%%~dpj
        for %%j in ("!JAVA_DIR!..") do set JAVA_HOME=%%~fj
        goto :javaFound
    )
    
    rem Try common Java installation locations
    if exist "C:\Program Files\Java\jdk*" (
        for /d %%d in ("C:\Program Files\Java\jdk*") do (
            set JAVA_HOME=%%d
            goto :javaFound
        )
    )
    
    if exist "C:\Program Files (x86)\Java\jdk*" (
        for /d %%d in ("C:\Program Files (x86)\Java\jdk*") do (
            set JAVA_HOME=%%d
            goto :javaFound
        )
    )
    
    rem Check registry for Java installation
    for /f "tokens=2*" %%a in ('reg query "HKLM\SOFTWARE\JavaSoft\Java Development Kit" /s /v JavaHome 2^>nul ^| findstr JavaHome') do (
        set JAVA_HOME=%%b
        goto :javaFound
    )
    
    echo **********************************************************************
    echo * ERROR: java could not be found. Please install the JRE or JDK.     *
    echo * If it is already installed, configure the system variables PATH or *
    echo * JAVA_HOME appropriately.                                           *
    echo **********************************************************************
    exit /b 1
)

:javaFound
echo Found JAVA_HOME: !JAVA_HOME! 1>&2
rem Use environment variable JAVA_EXECUTABLE if set, otherwise default to java.exe
if "!JAVA_EXECUTABLE!"=="" (
    set JAVA_EXECUTABLE=java.exe
)
set JAVA_CMD=!JAVA_HOME!\bin\!JAVA_EXECUTABLE!

if not exist "!JAVA_CMD!" (
    echo Error: Java executable not found at !JAVA_CMD! 1>&2
    exit /b 1
)
exit /b 0

:muleTroubleshooting
set LOCAL_OUTPUT_DIR=%~1

rem Setup paths using original script directory captured before shifts
set SCRIPT_DIR=!SCRIPT_DIR_ORIGINAL!
set JAR_PATH=!SCRIPT_DIR!\mule-troubleshooting-plugin-client-1.0.1-jar-with-dependencies.jar
set MAIN_CLASS=com.mulesoft.mule.troubleshooting.client.Client
set LOGS_DIR_SYSTEM_PROPERTY=-Ddiag.logs.dir=!SCRIPT_DIR!\..\logs

rem Rebuild arguments for Java application
rem Avoid shift entirely - use parameters directly
set JAVA_ARGS=--output "!LOCAL_OUTPUT_DIR!"
rem Add remaining args if they exist (%2 contains REMAINING_ARGS)
if not "%~2"=="" set JAVA_ARGS=!JAVA_ARGS! %~2

echo Calling mule-troubleshooting-plugin-client with args: !JAVA_ARGS! 1>&2

rem Check for debug option in arguments
echo !JAVA_ARGS! | findstr /c:"--debug" >nul
if !errorlevel! equ 0 (
    rem Remove --debug from JAVA_ARGS for execution
    set JAVA_ARGS=!JAVA_ARGS:--debug=!
    "!JAVA_CMD!" "!LOGS_DIR_SYSTEM_PROPERTY!" -agentlib:jdwp=transport=dt_socket,server=y,suspend=y,address=5005 -cp "!JAR_PATH!" "!MAIN_CLASS!" "!SCRIPT_DIR!" !JAVA_ARGS!
) else (
    "!JAVA_CMD!" "!LOGS_DIR_SYSTEM_PROPERTY!" -cp "!JAR_PATH!" "!MAIN_CLASS!" "!SCRIPT_DIR!" !JAVA_ARGS!
)
exit /b !errorlevel!

:createZipFile
set SOURCE_DIR=%1
set ZIP_PATH=%2

rem Use PowerShell to create zip file
powershell -command "Compress-Archive -Path '%SOURCE_DIR%\*' -DestinationPath '%ZIP_PATH%' -Force" >nul 2>&1
if !errorlevel! equ 0 (
    echo Diagnostic zip file created: !ZIP_PATH! 1>&2
) else (
    echo Error: Failed to create zip file !ZIP_PATH! 1>&2
    exit /b 1
)
exit /b 0
