package org.mule.tooling.browser.preference;

import static java.util.Arrays.stream;

import org.apache.commons.lang.StringUtils;
import org.eclipse.core.runtime.preferences.IEclipsePreferences;
import org.eclipse.core.runtime.preferences.InstanceScope;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.layout.GridDataFactory;
import org.eclipse.jface.layout.GridLayoutFactory;
import org.eclipse.jface.preference.FieldEditorPreferencePage;
import org.eclipse.jface.preference.RadioGroupFieldEditor;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.ShellAdapter;
import org.eclipse.swt.events.ShellEvent;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.graphics.Rectangle;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.IWorkbench;
import org.eclipse.ui.IWorkbenchPreferencePage;
import org.mule.tooling.browser.BrowserPlugin;
import org.mule.tooling.browser.BrowserProxy;
import org.mule.tooling.browser.BrowserUtils;
import org.mule.tooling.browser.BrowserUtils.Metric;
import org.mule.tooling.browser.IBrowser;

import com.google.common.base.CaseFormat;

public class BrowserPreferencePage extends FieldEditorPreferencePage implements IWorkbenchPreferencePage {

    public final static String BROWSER_PREFERENCE_KEY = "browser.preference.key";

    public BrowserPreferencePage() {
        super(GRID);
    }

    @Override
    public void init(IWorkbench workbench) {

        setPreferenceStore(BrowserPlugin.getDefault().getPreferenceStore());
        setDescription("Browser Settings");
    }

    @Override
    protected void createFieldEditors() {
        Group preview = new Group(getFieldEditorParent(), SWT.NONE);
        GridDataFactory.fillDefaults().grab(true, false).applyTo(preview);
        GridLayoutFactory.swtDefaults().applyTo(preview);
        final BrowserPreference[] preferences = BrowserPreference.values();
        BrowserPreferenceInitializer.getDefaultPreference(); // initialize disabled state
        final String[][] labelAndValues = stream(preferences).sorted().map(pref -> new String[] {
                StringUtils.capitalize(CaseFormat.UPPER_UNDERSCORE.to(CaseFormat.LOWER_HYPHEN, pref.name())) + (pref.isDisabled() ? "\n(" + pref.getDisabledCause() + ")" : ""),
                pref.name() }).toArray(String[][]::new);
        RadioGroupFieldEditor editor = new RadioGroupFieldEditor(BROWSER_PREFERENCE_KEY, "Default Browser", labelAndValues.length, labelAndValues, preview);
        addField(editor);
        if (stream(preferences).anyMatch(BrowserPreference::isDisabled)) {
            editor.setEnabled(false, preview);
        }

        Button previewButton = new Button(preview, SWT.NONE);
        previewButton.setText("Preview");
        previewButton.addSelectionListener(new SelectionAdapter() {

            @Override
            public void widgetSelected(SelectionEvent e) {
                createBrowserPreview(previewButton, editor);
            }
        });
    }

    protected void createBrowserPreview(Button previewButton, RadioGroupFieldEditor editor) {
        Shell shell = new Shell(getShell(), SWT.SHEET | SWT.SHELL_TRIM);
        shell.setLayout(new GridLayout());
        GridLayoutFactory.fillDefaults().applyTo(shell);

        int width = 600;
        int height = 400;

        shell.setSize(width, height);
        shell.setLocation(findLocation(previewButton, height));

        Composite owner = new Composite(shell, SWT.NONE);
        GridDataFactory.fillDefaults().grab(true, true).minSize(width, height).applyTo(owner);
        GridLayoutFactory.fillDefaults().margins(5, 5).applyTo(owner);

        final BrowserPreference browserPreference = BrowserPreference.valueOf(editor.getSelectionValue());
        IBrowser browser = null;
        switch (browserPreference) {
        case BUILT_IN:
            try {
                browser = BrowserProxy.wrap(BrowserProxy.createChromiumBrowser(owner));
            } catch (Throwable e) {
                BrowserPlugin.logError("Failed to preview browser " + BrowserPreference.BUILT_IN.name(), e);
            }
            break;
        case NATIVE:
            try {
                browser = BrowserProxy.wrap(BrowserProxy.createNativeBrowser(owner));
            } catch (Throwable e) {
                BrowserPlugin.logError("Failed to preview browser " + BrowserPreference.NATIVE.name(), e);
            }
            break;
        }
        if (browser == null) {
            MessageDialog.openInformation(getFieldEditorParent().getShell(), "Failed to create Browser",
                    "This configuration is not working, will fallback to other configuration in order to succeed");
            return;
        }
        browser.setUrl("https://www.mulesoft.com/");
        GridDataFactory.fillDefaults().grab(true, true).minSize(width, height).applyTo(browser.getComposite());

        shell.open();
        shell.addShellListener(new ShellAdapter() {

            @Override
            public void shellClosed(ShellEvent e) {
                super.shellClosed(e);
                if (!shell.isDisposed()) {
                    shell.dispose();
                }
            }

            @Override
            public void shellDeactivated(ShellEvent e) {
                super.shellDeactivated(e);
                if (!shell.isDisposed()) {
                    shell.dispose();
                }
            }
        });
    }

    private Point findLocation(Control control, int shellHeight) {
        Point location = getControlLocation(control);
        Rectangle bounds = control.getBounds();

        return new Point(location.x, location.y + bounds.height);
    }

    private Point getControlLocation(Control control) {
        return control.getParent().toDisplay(control.getLocation());
    }

    public static IEclipsePreferences getPreferences() {
        return InstanceScope.INSTANCE.getNode(BrowserPlugin.PLUGIN_ID);
    }

    public static BrowserPreference getBrowserPreference() {
        return BrowserPreference.valueOf(getPreferences().get(BROWSER_PREFERENCE_KEY, BrowserPreferenceInitializer.getDefaultPreference()));
    }

    public static void switchPreference(BrowserPreference browserPreference) {
        setPreference(browserPreference.equals(BrowserPreference.NATIVE) ? BrowserPreference.NATIVE : BrowserPreference.BUILT_IN);
        BrowserUtils.sendMetric(Metric.PREFERENCE_CHANGED.get());
    }

    public static void setPreference(BrowserPreference preference) {
        getPreferences().put(BROWSER_PREFERENCE_KEY, preference.name());
    }
}
