package org.mule.tooling.browser;

import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.function.BiFunction;
import java.util.function.Supplier;

import org.mule.tooling.browser.preference.BrowserPreferencePage;
import org.mule.tooling.core.analytics.AnalyticsRecordsFactory;

public class BrowserUtils {

    private static final String FUNCTION_PROPERTY_KEY = "function";
    private static final String URL_PROPERTY_KEY = "url";
    private static final String BROWSER_PROPERTY_KEY = "browser";

    public static void sendMetric(Metric metric) {
        final AnalyticsRecordsFactory event = AnalyticsRecordsFactory.single(metric.getEventId());
        metric.getProperties().forEach((propKey, propValue) -> {
            event.addMetadata(propKey, propValue);
        });
        event.track();
    }

    public static Map<String, String> withCurrentBrowser() {
        Map<String, String> props = new HashMap<String, String>();
        props.put(BROWSER_PROPERTY_KEY, BrowserPreferencePage.getBrowserPreference().name());
        return props;
    }

    public static Map<String, String> withContext(String function, String url) {
        final Map<String, String> props = withCurrentBrowser();
        props.put(URL_PROPERTY_KEY, url);
        props.put(FUNCTION_PROPERTY_KEY, function);
        return props;
    }

    public static class Metric {

        private static final String NO_BROWSER_AVAILABLE_EVENT = "NO BROWSER AVAILABLE";
        private static final String BROWSER_FUNCTION_EXEC_EVENT = "BROWSER_FUNCTION_EXEC";
        private static final String BROWSER_PREFERENCE_CHANGED_EVENT = "BROWSER_PREFERENCE_CHANGED";
        private static final String BROWSER_OPEN_FAIL_EVENT = "BROWSER_OPEN_FAIL";
        private static final String BROWSER_SSO_LOGIN_EVENT = "BROWSER_SSO_LOGIN";
        private static final String BROWSER_LOGIN_EVENT = "BROWSER_LOGIN";
        private static final String BROWSER_OPEN_EVENT = "BROWSER_OPEN";

        public final static Supplier<Metric> ON_OPEN = () -> new Metric(BROWSER_OPEN_EVENT, BrowserUtils.withCurrentBrowser());
        public final static Supplier<Metric> ON_LOGIN = () -> new Metric(BROWSER_LOGIN_EVENT, BrowserUtils.withCurrentBrowser());
        public final static Supplier<Metric> ON_SSO_LOGIN = () -> new Metric(BROWSER_SSO_LOGIN_EVENT, BrowserUtils.withCurrentBrowser());
        public final static Supplier<Metric> OPEN_FAIL = () -> new Metric(BROWSER_OPEN_FAIL_EVENT, BrowserUtils.withCurrentBrowser());
        public final static Supplier<Metric> NO_BROWSER_AVAILABLE = () -> new Metric(NO_BROWSER_AVAILABLE_EVENT, Collections.emptyMap());
        public final static Supplier<Metric> PREFERENCE_CHANGED = () -> new Metric(BROWSER_PREFERENCE_CHANGED_EVENT, BrowserUtils.withCurrentBrowser());
        public final static BiFunction<String, String, Metric> FUNCTION_EXEC = (function, url) -> new Metric(BROWSER_FUNCTION_EXEC_EVENT, BrowserUtils.withContext(function, url));

        final Map<String, String> properties;
        final String eventId;

        Metric(String eventId, Map<String, String> properties) {
            this.eventId = eventId;
            this.properties = properties;
        }

        public String getEventId() {
            return eventId;
        }

        public Map<String, String> getProperties() {
            return properties;
        }
    }

}