package org.mule.tooling.browser;

import java.lang.reflect.Proxy;

import org.eclipse.swt.SWT;
import org.eclipse.swt.SWTError;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Menu;
import org.mule.tooling.browser.BrowserUtils.Metric;
import org.mule.tooling.browser.impl.BrowserInvocationHandler;
import org.mule.tooling.browser.impl.NoBrowser;
import org.mule.tooling.browser.preference.BrowserPreference;
import org.mule.tooling.browser.preference.BrowserPreferencePage;

public class BrowserProxy {

    public static IBrowser createBrowser(Composite parent) {
        final BrowserPreference browserPreference = BrowserPreferencePage.getBrowserPreference();
        Object target;
        try {
            switch (browserPreference) {
            case BUILT_IN:
                target = createChromiumBrowser(parent);
                break;
            case NATIVE:
            default:
                target = createNativeBrowser(parent);
                break;
            }
            BrowserUtils.sendMetric(Metric.ON_OPEN.get());
        } catch (Throwable e) { // fallback to other alternative
            BrowserUtils.sendMetric(Metric.OPEN_FAIL.get());
            String browser = "";
            try {
                if (browserPreference.equals(BrowserPreference.BUILT_IN)) {
                    browser = BrowserPreference.NATIVE.name();
                    target = createNativeBrowser(parent);
                } else {
                    browser = BrowserPreference.BUILT_IN.name();
                    target = createChromiumBrowser(parent);
                }
                // Should not try never again with failing browser preference, unless user changes it again.
                BrowserPlugin.logError("Failed to create browser, retrying with '" + browser + "' browser. Preference will be set to use this browser", e);
                BrowserPreferencePage.switchPreference(browserPreference);

            } catch (Exception browserException) {
                BrowserUtils.sendMetric(Metric.NO_BROWSER_AVAILABLE.get());
                BrowserPlugin.logError("Failed to create browser. No browser is available", browserException);
                target = new NoBrowser(parent);
                return (IBrowser) target;
            }

        }
        return wrap(target);
    }

    public static IBrowser wrap(Object target) {
        return (IBrowser) Proxy.newProxyInstance(//
                BrowserProxy.class.getClassLoader(), //
                new Class[] { IBrowser.class }, //
                new BrowserInvocationHandler(target));
    }

    public static com.equo.chromium.swt.Browser createChromiumBrowser(Composite parent) {
        // This is needed in order to avoid CORS from blocking requests done from Studio's API Console
        if (System.getProperty("chromium.args") == null) {
            System.setProperty("chromium.args", "--disable-web-security");
        }

        com.equo.chromium.swt.Browser browser = new com.equo.chromium.swt.Browser(parent, SWT.NONE);
        browser.setMenu(new Menu(parent));
        return browser;
    }

    @SuppressWarnings("deprecation")
    public static org.eclipse.swt.browser.Browser createNativeBrowser(Composite composite) {
        org.eclipse.swt.browser.Browser browser;
        try {
            browser = new org.eclipse.swt.browser.Browser(composite, SWT.MOZILLA);
            browser.setMenu(new Menu(composite));
            return browser;
        } catch (SWTError webkitError) {
            try {
                browser = new org.eclipse.swt.browser.Browser(composite, SWT.WEBKIT);
                browser.setMenu(new Menu(composite));
                return browser;
            } catch (SWTError mozillaError) {
                try {
                    browser = new org.eclipse.swt.browser.Browser(composite, SWT.NONE);
                    browser.setMenu(new Menu(composite));
                    return browser;
                } catch (SWTError browserError) {
                    throw browserError;
                }
            }
        }
    }

    public static void clearSessions() {
        final BrowserPreference browserPreference = BrowserPreferencePage.getBrowserPreference();
        switch (browserPreference) {
        case BUILT_IN:
            com.equo.chromium.swt.Browser.clearSessions();
            break;
        case NATIVE:
            org.eclipse.swt.browser.Browser.clearSessions();
            break;
        }
    }

    public static void setCookie(String value, String url) {
        final BrowserPreference browserPreference = BrowserPreferencePage.getBrowserPreference();
        switch (browserPreference) {
        case BUILT_IN:
            com.equo.chromium.swt.Browser.setCookie(value, url);
            break;
        case NATIVE:
            org.eclipse.swt.browser.Browser.setCookie(value, url);
            break;
        }
    }

    public static String getCookie(String value, String url) {
        final BrowserPreference browserPreference = BrowserPreferencePage.getBrowserPreference();
        switch (browserPreference) {
        case BUILT_IN:
            return com.equo.chromium.swt.Browser.getCookie(value, url);
        case NATIVE:
            return org.eclipse.swt.browser.Browser.getCookie(value, url);
        }
        return null;
    }

}
